<?php
namespace Avayemarketing\AvaSMS\SMS;

use Avayemarketing\AvaSMS\Logging\Logger;
use Avayemarketing\AvaSMS\Settings\Options;

if (!defined('ABSPATH')) exit;

final class SmsModule
{
    public function register_hooks(): void
    {
        add_filter('avasms_send_sms', [$this, 'filter_send_sms'], 10, 3);
    }

    /**
     * Compatibility filter used by admin AJAX and other modules.
     * Return bool for backward compatibility.
     *
     * @param mixed  $default
     * @param string $to
     * @param string $text
     * @return bool
     */
    public function filter_send_sms($default, $to, $text): bool
    {
        $result = $this->send((string)$to, (string)$text);

        if (!empty($result['success'])) return true;

        Logger::log('error', 'sms', 'ارسال پیامک ناموفق بود.', [
            'to' => (string)$to,
            'error_code' => $result['error_code'] ?? null,
            'provider' => $result['provider'] ?? null,
            'provider_response' => $result['provider_response'] ?? null,
        ]);

        return false;
    }

    public function send(string $to, string $message, array $args = []): array
    {
        $to = Utils::normalize_iran_mobile($to);
        if ($to === '') {
            return [
                'success' => false,
                'message' => 'شماره موبایل نامعتبر است.',
                'provider' => null,
                'provider_response' => null,
                'error_code' => 'invalid_mobile',
            ];
        }

        $message = trim($message);
        if ($message === '') {
            return [
                'success' => false,
                'message' => 'متن پیام خالی است.',
                'provider' => null,
                'provider_response' => null,
                'error_code' => 'empty_message',
            ];
        }

        $gw = Options::gateways(); // normalized structure
        $active = (string)($gw['active_gateway'] ?? 'kavenegar');

        $driver = GatewayFactory::make($active, $gw);
        if (!$driver->is_configured()) {
            return [
                'success' => false,
                'message' => 'تنظیمات درگاه پیامکی کامل نیست.',
                'provider' => $driver->key(),
                'provider_response' => null,
                'error_code' => 'not_configured',
            ];
        }

        $res = $driver->send($to, $message, $args);
        $res['provider'] = $driver->key();

        return $res;
    }
}
