<?php
namespace Avayemarketing\AvaSMS\Settings;

if (!defined('ABSPATH')) exit;

final class Options
{
    public const OPT_CAMPAIGNS = 'avasms_campaigns_options';

    /**
     * New gateway option key (per spec).
     */
    public const OPT_GATEWAYS  = 'avasms_gateways';

    /**
     * Legacy key (v0.6.x)
     */
    public const OPT_GATEWAYS_LEGACY = 'avasms_sms_gateways';

    public const OPT_OTP       = 'avasms_otp_options';
    public const OPT_WOO       = 'avasms_woo_sms_options';

    public static function get(string $option_name, array $default = []): array
    {
        $val = get_option($option_name, []);
        return is_array($val) ? array_merge($default, $val) : $default;
    }

    public static function update(string $option_name, array $value): bool
    {
        return update_option($option_name, $value, false);
    }

    /**
     * Normalized gateways settings array:
     * [
     *   active_gateway => kavenegar|ippanel|farazsms|melli_sms,
     *   kavenegar => [...],
     *   ippanel   => [...],
     *   farazsms  => [...],
     *   melli_sms => [...],
     * ]
     */
    public static function gateways(): array
    {
        $default = [
            'active_gateway' => 'kavenegar',

            'kavenegar' => [
                'api_key'       => '',
                'sender_line'   => '',
                'template_code' => '',
                'use_pattern'   => 0,
                'sample_text'   => 'کد ورود شما: {code}',
            ],

            'ippanel' => [
                'api_key'       => '',
                'sender_line'   => '',
                'template_code' => '',
                'use_pattern'   => 0,
                'endpoint'      => 'https://api2.ippanel.com/api/v1/sms/send/webservice/single',
                'sample_text'   => 'پیام تست AvaSMS',
            ],

            'farazsms' => [
                'username'      => '',
                'password'      => '',
                'sender_line'   => '',
                'template_code' => '',
                'use_pattern'   => 0,
                'endpoint'      => '',
                'sample_text'   => 'پیام تست AvaSMS',
            ],

            'melli_sms' => [
                'username'      => '',
                'password'      => '',
                'sender_line'   => '',
                'template_code' => '',
                'use_pattern'   => 0,
                'endpoint'      => '',
                'sample_text'   => 'پیام تست AvaSMS',
            ],
        ];

        $val = get_option(self::OPT_GATEWAYS, null);

        // migrate from legacy if needed
        if (!is_array($val) || empty($val)) {
            $legacy = get_option(self::OPT_GATEWAYS_LEGACY, null);
            if (is_array($legacy) && !empty($legacy)) {
                $val = self::migrate_legacy_gateways($legacy);
                update_option(self::OPT_GATEWAYS, $val, false);
            }
        }

        $val = is_array($val) ? array_merge($default, $val) : $default;

        // allow backward compatible key "active"
        if (!empty($val['active']) && empty($val['active_gateway'])) {
            $val['active_gateway'] = sanitize_key((string)$val['active']);
        }

        // sanitize active
        $active = sanitize_key((string)($val['active_gateway'] ?? 'kavenegar'));
        if (!in_array($active, ['kavenegar','ippanel','farazsms','melli_sms','melipayamak'], true)) $active = 'kavenegar';
        if ($active === 'melipayamak') $active = 'melli_sms';
        $val['active_gateway'] = $active;

        return $val;
    }

    private static function migrate_legacy_gateways(array $legacy): array
    {
        // legacy structure used 'active' key and 'melipayamak'
        $out = [
            'active_gateway' => sanitize_key((string)($legacy['active'] ?? 'kavenegar')),
            'kavenegar' => $legacy['kavenegar'] ?? [],
            'ippanel'   => $legacy['ippanel'] ?? [],
            'farazsms'  => $legacy['farazsms'] ?? [],
            'melli_sms' => $legacy['melipayamak'] ?? ($legacy['melli_sms'] ?? []),
        ];
        if ($out['active_gateway'] === 'melipayamak') $out['active_gateway'] = 'melli_sms';
        return $out;
    }

    public static function otp(): array
    {
        return self::get(self::OPT_OTP, [
            'enabled' => 1,
            'length'  => 5,
            'expiry_seconds' => 120,
            'max_send_per_hour' => 10,
            'max_verify_attempts' => 8,
            'lock_minutes' => 30,
            'message_template' => 'کد ورود شما: {code}',

            // Login identifier
            'allow_login_with_email' => 1,

            // Cloudflare Turnstile (optional)
            'turnstile_enabled' => 0,
            'turnstile_site_key' => '',
            'turnstile_secret_key' => '',
        ]);
    }

    public static function woo(): array
    {
        return self::get(self::OPT_WOO, [
            'enabled' => 1,

            'customer_enabled' => 1,
            'customer_on_new_order' => 1,
            'customer_on_status_change' => 1,
            'customer_statuses' => [
                'processing' => 1,
                'completed'  => 1,
                'on-hold'    => 0,
                'cancelled'  => 0,
                'refunded'   => 0,
                'failed'     => 0,
            ],

            'admin_enabled' => 1,
            'admin_phones'  => '',
            'admin_on_new_order' => 1,
            'admin_on_status_change' => 1,
            'admin_statuses' => [
                'processing' => 1,
                'completed'  => 1,
                'on-hold'    => 1,
                'cancelled'  => 1,
                'refunded'   => 1,
                'failed'     => 1,
            ],

            'tpl_customer_new_order' => 'سفارش شما ثبت شد. شماره سفارش: {order_id} | مبلغ: {price}',
            'tpl_customer_status'    => 'وضعیت سفارش {order_id} به {status} تغییر کرد.',
            'tpl_admin_new_order'    => 'سفارش جدید ثبت شد. شماره سفارش: {order_id} | مبلغ: {price}',
            'tpl_admin_status'       => 'وضعیت سفارش {order_id} به {status} تغییر کرد.',
        ]);
    }

    public static function campaigns(): array
    {
        return self::get(self::OPT_CAMPAIGNS, [
            'batch_size' => 20,
            'top_buyers_limit' => 50,
            'birthday_meta_key' => 'birthdate',
        ]);
    }
}
