<?php
namespace Avayemarketing\AvaSMS\Logging;

if (!defined('ABSPATH')) exit;

final class Logger
{
    private const TABLE = 'avasms_logs';

    public function register_hooks(): void
    {
        // reserved
    }

    public function install(): void
    {
        global $wpdb;
        $table = self::table_name();
        $charset = $wpdb->get_charset_collate();

        $sql = "CREATE TABLE {$table} (
            id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
            created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
            level VARCHAR(20) NOT NULL,
            context VARCHAR(50) NOT NULL,
            message TEXT NOT NULL,
            data LONGTEXT NULL,
            PRIMARY KEY (id),
            KEY level (level),
            KEY context (context),
            KEY created_at (created_at)
        ) {$charset};";

        require_once ABSPATH . 'wp-admin/includes/upgrade.php';
        dbDelta($sql);
    }

    public static function table_name(): string
    {
        global $wpdb;
        return $wpdb->prefix . self::TABLE;
    }

    /**
     * @param string $level info|warning|error
     * @param string $context e.g. sms-single|campaign|test
     * @param string $message
     * @param array $data
     */
    public static function log(string $level, string $context, string $message, array $data = []): void
    {
        global $wpdb;

        $level = sanitize_key($level);
        if (!in_array($level, ['info','warning','error'], true)) $level = 'info';

        $context = sanitize_key($context);
        $message = wp_kses_post($message);

        $encoded = !empty($data) ? wp_json_encode($data, JSON_UNESCAPED_UNICODE) : null;

        $wpdb->insert(self::table_name(), [
            'created_at' => current_time('mysql'),
            'level'      => $level,
            'context'    => $context,
            'message'    => $message,
            'data'       => $encoded,
        ], ['%s','%s','%s','%s','%s']);

        // Optional: email admin on high severity errors
        if ($level === 'error' && !empty($data['severity']) && $data['severity'] === 'high') {
            $to = get_option('admin_email');
            if ($to) {
                $subject = 'AvaSMS: خطای ارسال پیامک';
                $body = $message . "\n\n" . (is_string($encoded) ? $encoded : '');
                wp_mail($to, $subject, $body);
            }
        }
    }

    public static function recent(int $limit = 200, ?string $level = null): array
    {
        global $wpdb;
        $limit = max(1, min(1000, $limit));

        $where = '';
        if ($level) {
            $level = sanitize_key($level);
            if (in_array($level, ['info','warning','error'], true)) {
                $where = $wpdb->prepare("WHERE level=%s", $level);
            }
        }

        $sql = "SELECT * FROM " . self::table_name() . " {$where} ORDER BY id DESC LIMIT {$limit}";
        $rows = $wpdb->get_results($sql, ARRAY_A);
        return is_array($rows) ? $rows : [];
    }

    public static function clear_older_than_days(int $days): int
    {
        global $wpdb;
        $days = max(1, min(3650, $days));
        $cutoff = gmdate('Y-m-d H:i:s', time() - ($days * DAY_IN_SECONDS));
        return (int)$wpdb->query($wpdb->prepare("DELETE FROM " . self::table_name() . " WHERE created_at < %s", $cutoff));
    }
}
