<?php
namespace Avayemarketing\AvaSMS\User;

if (!defined('ABSPATH')) exit;

/**
 * Adds a Birthdate meta field (avasms_birthdate) to:
 * - WP Admin user profile (edit user)
 * - Default WP registration form (optional)
 *
 * Stored format: YYYY-MM-DD (same as <input type="date">).
 */
final class BirthdateField
{
    public const META_KEY = 'avasms_birthdate';

    public function register_hooks(): void
    {
        // Admin profile screens
        add_action('show_user_profile', [$this, 'render_admin_field']);
        add_action('edit_user_profile', [$this, 'render_admin_field']);

        add_action('personal_options_update', [$this, 'save_admin_field']);
        add_action('edit_user_profile_update', [$this, 'save_admin_field']);

        // Default WP registration form (wp-login.php?action=register)
        add_action('register_form', [$this, 'render_register_field']);
        add_action('user_register', [$this, 'save_register_field'], 10, 1);
    }

    public function render_admin_field($user): void
    {
        if (!($user instanceof \WP_User)) return;
        if (!current_user_can('edit_user', $user->ID)) return;

        $val = (string) get_user_meta($user->ID, self::META_KEY, true);
        ?>
        <h2><?php echo esc_html__('اطلاعات تکمیلی', 'avasms'); ?></h2>
        <table class="form-table" role="presentation">
            <tr>
                <th><label for="avasms_birthdate"><?php echo esc_html__('تاریخ تولد', 'avasms'); ?></label></th>
                <td>
                    <input type="date"
                           name="avasms_birthdate"
                           id="avasms_birthdate"
                           value="<?php echo esc_attr($val); ?>"
                           class="regular-text" />
                    <p class="description"><?php echo esc_html__('فرمت ذخیره‌سازی: YYYY-MM-DD', 'avasms'); ?></p>
                </td>
            </tr>
        </table>
        <?php
    }

    public function save_admin_field(int $user_id): void
    {
        if (!current_user_can('edit_user', $user_id)) return;
        if (!isset($_POST['avasms_birthdate'])) return;

        $bd = sanitize_text_field(wp_unslash($_POST['avasms_birthdate']));
        if ($bd === '' || preg_match('/^\d{4}-\d{2}-\d{2}$/', $bd)) {
            update_user_meta($user_id, self::META_KEY, $bd);
        }
    }

    public function render_register_field(): void
    {
        $val = '';
        if (!empty($_POST['avasms_birthdate'])) {
            $val = sanitize_text_field(wp_unslash($_POST['avasms_birthdate']));
        }
        ?>
        <p>
            <label for="avasms_birthdate"><?php echo esc_html__('تاریخ تولد', 'avasms'); ?><br/>
                <input type="date" name="avasms_birthdate" id="avasms_birthdate" class="input" value="<?php echo esc_attr($val); ?>" />
            </label>
        </p>
        <?php
    }

    public function save_register_field(int $user_id): void
    {
        if (!isset($_POST['avasms_birthdate'])) return;
        $bd = sanitize_text_field(wp_unslash($_POST['avasms_birthdate']));
        if ($bd === '' || preg_match('/^\d{4}-\d{2}-\d{2}$/', $bd)) {
            update_user_meta($user_id, self::META_KEY, $bd);
        }
    }
}
